/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var app = {
  deviceReady: false,
  validPackets: 0,
  invalidPackets: 0,
  packetCounts: {
    heartbeat: 0,
    init: 0,
    uplink: 0,
    hat: 0,
    ownship: 0,
    ownshipGeoAlt: 0,
    traffic: 0,
    basic: 0,
    long: 0,
    short1090: 0,
    long1090: 0,
    uavionixConfigReport: 0,
    uavionixStatusReport: 0,
    uavionixTowerStatusReport: 0,
    uavionixDeviceStartup: 0,
    uavionixGeofence: 0
  },
  logoClicked: 0,
  logoClickTimeout: null,
  trackingClick: false,
  advancedDisplayed: false,
  view: 'configuration', // options are 'configuration' and 'monitor'

  ownshipReceived: false,
  ownship: null,
  ownshipGeoAltReceived: false,
  ownshipGeoAlt: null,

  configReportReceived: false,
  configReport: null,

  statusReportReceived: false,
  statusReport: null,

  deliveredConfigurationWarning: false,

  // IP address of Echo device
  deviceIp: "192.168.4.1",

  connectionTimeout: null,
  // RCB TODO validate if needed for Echo devices
  // Periodically send packet to device to enter GDL 90 mode
  enterGDL90ModeTimer: null,
  // Ownship received, wait for status with remainder of config
  ownshipToStatusTimer: null,
  // Solicit Status Report timer, used for certain devices
  solicitStatusReportTimer: null,

  geofenceData: null,
  geofencePacketReceived: null,
  
  // Application Constructor
  initialize: function () {
    this.updateMonitorInfo();
    this.updateFirmwareVersionInfo();
    document.getElementById("appVersion").innerHTML = "App Version: " + require('electron').remote.app.getVersion();
    document.title = document.title + " (" + require('electron').remote.app.getVersion() + ")";
	  this.onDeviceReady();
    //this.bindEvents();
  },
  // Bind Event Listeners
  //
  // Bind any events that are required on startup. Common events are:
  // 'load', 'deviceready', 'offline', and 'online'.
  bindEvents: function () {
    document.addEventListener('deviceready', this.onDeviceReady.bind(this), false);
  },
  // deviceready Event Handler
  //
  // The scope of 'this' is the event. In order to call the 'receivedEvent'
  // function, we must explicitly call 'app.receivedEvent(...);'
  onDeviceReady: function () {
    this.deviceReady = true;

    this.platformSpecificSetup();

    // Delay execution until ready with debugger
    //window.alert("Waiting for debugger");

    // Set up additional listeners
    document.addEventListener('pause', this.onPause.bind(this), false);
    document.addEventListener('resume', this.onResume.bind(this), false);

    app.receivedEvent('devicestatus', 'deviceready');
    
    serial.initialize(null, this.dataReceived.bind(this), ui.populateComPortsCallback.bind(this));
    ui.initialize();
    gdl90.initialize();
    
    // Not necessary for Echo but may be useful
    //this.enterGDL90ModeTimer = setInterval(this.sendGDL90Wakeup.bind(this), 1000);
    // Only use for selected devices
    if (this.solicitStatusReportTimer == null && ui.currentDevice && ui.currentDevice.solicitStatusReport) {
      this.solicitStatusReportTimer = setInterval(this.sendSolicitStatusReport.bind(this), 1000);
    }

  },
  platformSpecificSetup: function () {
    if (typeof device != "undefined" && device.platform == "Android") {
      document.getElementById("skyBeaconMessageSpan").innerHTML = "To configure your skyBeacon, please use the <a href='market://details?id=com.uavionix.skybeacon' rel='external'>uAvionix skyBeacon Installer</a><br>";
    } else if (typeof device != "undefined" && device.platform == "iOS") {
      document.getElementById("skyBeaconMessageSpan").innerHTML = "To configure your skyBeacon, please use the <a href='itms-apps://itunes.apple.com/app/id1291016785'>uAvionix skyBeacon Installer</a><br>";
    } else {
      document.getElementById("skyBeaconMessageSpan").innerHTML = "To configure your skyBeacon, please use the uAvionix skyBeacon Installer<br>";
    }
  },
  // On iOS this function cannot call any native calls until resume
  // "resign" event is available as an alternative to handle background lock situations
  onPause: function () {
    if (this.enterGDL90ModeTimer != null) {
      clearTimeout(this.enterGDL90ModeTimer);
      this.enterGDL90ModeTimer = null;
    }
    if (this.solicitStatusReportTimer != null) {
      clearTimeout(this.solicitStatusReportTimer);
      this.solicitStatusReportTimer = null;
    }
    clearTimeout(this.connectionTimeout);
    this.connectionTimeout = null;
    clearTimeout(this.ownshipToStatusTimer);
    this.ownshipToStatusTimer = null;
    this.deviceDisconnected();

    serial.deinitialize();
  },
  onResume: function () {
    while (!serial.isReadyToInitialize()) { }
    // Subsequent init doesn't require callback
    serial.initialize(this.deviceIp);
    if (this.enterGDL90ModeTimer == null) {
      // Disabled for Echo devices
      //this.enterGDL90ModeTimer = setInterval(this.sendGDL90Wakeup.bind(this), 1000);
    }
    if (this.solicitStatusReportTimer == null && ui.currentDevice && ui.currentDevice.solicitStatusReport) {
      this.solicitStatusReportTimer = setInterval(this.sendSolicitStatusReport.bind(this), 1000);
    }
  },
  // Called when we see the ping2020 device
  onDeviceConnected: function () {
    // Update the UI
    app.receivedEvent('devicestatus', 'deviceconnected');
    // Only use for selected devices
    if (this.solicitStatusReportTimer == null && ui.currentDevice && ui.currentDevice.solicitStatusReport) {
      this.solicitStatusReportTimer = setInterval(this.sendSolicitStatusReport.bind(this), 1000);
    } else if (this.solicitStatusReportTimer != null && ui.currentDevice && ui.currentDevice.solicitStatusReport) {
      // timer already running and should be
    } else if (this.solicitStatusReportTimer != null) {
      // timer already running and shouldn't be
      clearTimeout(this.solicitStatusReportTimer);
      this.solicitStatusReportTimer = null;
    }
  },
  // Update DOM on a Received Event
  receivedEvent: function (id, status) {
    var parentElement = document.getElementById(id);
    var listeningElement = parentElement.querySelector('.listening');
    var receivedElement = parentElement.querySelector('.received');
    var connectedElement = parentElement.querySelector('.connected');

    if (status == 'devicelistening') {
      // Waiting for Phonegap
      listeningElement.setAttribute('style', 'display:block;');
      receivedElement.setAttribute('style', 'display:none;');
      connectedElement.setAttribute('style', 'display:none;');
    } else if (status == 'deviceready') {
      // Phonegap up, waiting for GDL90 connection
      listeningElement.setAttribute('style', 'display:none;');
      receivedElement.setAttribute('style', 'display:block;');
      connectedElement.setAttribute('style', 'display:none;');
    } else if (status == 'deviceconnected') {
      // GDL90 connection validated
      listeningElement.setAttribute('style', 'display:none;');
      receivedElement.setAttribute('style', 'display:none;');
      connectedElement.setAttribute('style', 'display:block;');
    }

    //console.log('Received Event: ' + id + "/" + status);
  },
  sendGDL90Wakeup: function () {
    // Send a benign packet to enter GDL 90 mode on the device if necessary
    var gdl90Packet = gdl90.constructSetupV2(0xFF, 0xFF, 0xFFFF, 0xFF, 0xFF).buffer;
    serial.send(gdl90Packet, null);
  },
  sendSolicitStatusReport: function() {
    // Send a request for a status report
    var gdl90Packet = gdl90.constructSolicitStatusReport().buffer;
    serial.send(gdl90Packet, null);
  },
  deviceDisconnected: function() {
    // No longer receiving data from device
    this.connectionTimeout = null;
    this.ownshipReceived = false;
    this.ownship = null;
    this.ownshipGeoAltReceived = false;
    this.ownshipGeoAlt = null;
    this.configReportReceived = false;
    this.configReport = null;
    this.statusReportReceived = false;
    this.statusReport = null;
    this.deliveredConfigurationWarning = false;
    // Always enabled for echo
    //document.getElementById("tabHeader").style.display = "none";
    this.setView("configuration");
    this.updateMonitorInfo();
    this.updateFirmwareVersionInfo();
    this.resetPacketCounts();
    this.receivedEvent('devicestatus', 'deviceready');
    document.getElementById("deviceType").disabled = false;
    document.getElementById("configForm").reset();
  },
  toggleView: function () {
    if (this.view == "monitor") {
      this.setView("configuration");
    } else {
      this.setView("monitor");
    }
  },
  setView: function (newView) {
    if (newView == "monitor" || newView == "configuration") {
      this.view = newView;
      return; // VTU only has config
      if (this.view == "monitor") {
        document.getElementById("tabConfiguration").className = "headertab deselected";
        document.getElementById("tabMonitor").className = "headertab selected";
        document.getElementById("deviceconfiguration").setAttribute('style', 'display:none;');
        document.getElementById("devicemonitor").setAttribute('style', 'display:block;');
      } else {
        document.getElementById("tabConfiguration").className = "headertab selected";
        document.getElementById("tabMonitor").className = "headertab deselected";
        document.getElementById("devicemonitor").setAttribute('style', 'display:none;');
        document.getElementById("deviceconfiguration").setAttribute('style', 'display:block;');
      }
    }

  },
  // Determine if ownship has enough data to display monitor
  isOwnshipRich: function (ownship) {
    if (ownship.callsign.trim().length != 0 ||
      ownship.address != "000000" ||
      (ownship.latitude != "--" && ownship.longitude != "--") ||
      ownship.altitude != "--") {
      return true;
    } else {
      return false;
    }
  },
  dataReceived: function (data) {
    // Received data
    var packet = new Uint8Array(data);
    //console.log("Received data on " + result.socketId + ", " + packet.length + " from " + result.remoteAddress + ":" + result.remotePort);
    var validPacketArray = gdl90.validatePackets(packet);
    validPacketArray.forEach(function (validPacket) {
      if (validPacket.valid) {
        this.validPackets++;
        // Mark the device as connected
        this.onDeviceConnected();

        if (validPacket.type == gdl90.MESSAGE_ID_OWNSHIP) {
          this.packetCounts.ownship++;
          var oldOwnship = this.ownship;
          this.ownship = gdl90.decodeOwnship(validPacket.packet);
          if (!this.ownshipReceived && this.isOwnshipRich(this.ownship)) {
            // First ownship packet of this connection
            // If rich enough, set the view to the monitor and enable the tabs
            // TODO offer to fill data when this goes valid (GPS lock?)
            //this.setView('monitor');
            document.getElementById("tabHeader").style.display = "block";
            this.ownshipReceived = true;
          }
          // If configuration specific ownship details have been updated, then reload configuration
          if (oldOwnship == null ||
            (this.ownship.address.trim() != oldOwnship.address.trim()) ||
            (this.ownship.emitterCategory != oldOwnship.emitterCategory) ||
            (this.ownship.callsign.trim() != oldOwnship.callsign.trim())) {
            //navigator.notification.alert("Received updated ownship", null, "Configuration", "OK");
            // If we don't have a status packet yet, defer loading configuration - wait up to one second 
            // for the status packet to arrive
            // If we do have a status packet fill in the ownship info
            if (this.configReportReceived) {
              this.loadConfiguration();
            } else {
              this.ownshipToStatusTimer = setTimeout(this.waitForOwnshipAndStatus.bind(this), 1000);
            }
          }
        } else if (validPacket.type == gdl90.MESSAGE_ID_OWNSHIP_GEO_ALT) {
          this.packetCounts.ownshipGeoAlt++;
          this.ownshipGeoAlt = gdl90.decodeOwnshipGeoAlt(validPacket.packet);
          this.ownshipGeoAltReceived = true;
        } else if (validPacket.type == gdl90.MESSAGE_ID_HEARTBEAT) {
          this.packetCounts.heartbeat++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_INITIALIZATION) {
          this.packetCounts.init++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_UPLINK) {
          this.packetCounts.uplink++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_HAT) {
          this.packetCounts.hat++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_TRAFFIC) {
          this.packetCounts.traffic++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_BASIC) {
          this.packetCounts.basic++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_LONG) {
          this.packetCounts.long++;
        } else if (validPacket.type == gdl90.MESSAGE_ID_UAVIONIX) {
          if (validPacket.packet.length >= 4) {
            // Geofence
            if (validPacket.packet[1] == gdl90.UAVIONIX_SIGNATURE &&
              validPacket.packet[2] == gdl90.UA_SUBTYPE_GEOFENCE &&
              validPacket.packet[3] == 0x01) {
              this.packetCounts.uavionixGeofence++;
              if (this.geofencePacketReceived)
                this.geofencePacketReceived(validPacket.packet);
            }
            // Configuration report
            else if (validPacket.packet[1] == gdl90.UAVIONIX_SIGNATURE &&
              validPacket.packet[2] == gdl90.UAVIONIX_OEM_CONFIGURATION_REPORT) {

              // Configuration report v1

              // Clear the timer that is triggered when we receive our first ownship
              // and we are waiting for a status packet to determine if we have full
              // or partial info to populate the configuration screen
              if (this.ownshipToStatusTimer !== null) {
                clearTimeout(this.ownshipToStatusTimer);
                this.ownshipToStatusTimer = null;
              }
              // on receive of status packet we want to switch to the monitor view
              // and crossfill data?
              // update headertab to add monitor
              this.packetCounts.uavionixConfigReport++;
              var oldConfigReport = this.configReport;
              // Select appropriate version configuration report
              if (validPacket.packet[3] == 0x01) {
                this.configReport = gdl90.decodeConfigurationReport(validPacket.packet);
              } else if (validPacket.packet[3] == 0x02) {
                this.configReport = gdl90.decodeConfigurationReportV2(validPacket.packet);
              } else if (validPacket.packet[3] == 0x03) {
                this.configReport = gdl90.decodeConfigurationReportV3(validPacket.packet);
              } else if (validPacket.packet[3] == 0x04) {
                this.configReport = gdl90.decodeConfigurationReportV4(validPacket.packet);
              } else if (validPacket.packet[3] == 0x05) {
                this.configReport = gdl90.decodeConfigurationReportV5(validPacket.packet);
              } else if (validPacket.packet[3] >= 0x06) {
                this.configReport = gdl90.decodeConfigurationReportV6(validPacket.packet);
                
              }
              if (!this.configReportReceived) {
                this.configReportReceived = true;
                //this.setView('monitor');
              }
              // Compare to see if we have an updated status - use JSON stringify to test
              if ((oldConfigReport == null) ||
                (JSON.stringify(oldConfigReport) != JSON.stringify(this.configReport))) {
                //navigator.notification.alert("Received updated status", null, "Configuration", "OK");
                // If we don't have an ownship packet yet, defer loading configuration - wait up to one second 
                // for the ownship packet to arrive
                // If we do have a ownship packet fill in the configuration
                // RCB VTU-20 ICC doesn't send ownship, disable this check
                //if (this.ownshipReceived) {
                  this.loadConfiguration();
                //} else {
                  //this.ownshipToStatusTimer = setTimeout(this.waitForOwnshipAndStatus.bind(this), 1000);
                //}
              }

              document.getElementById("tabHeader").style.display = "block";
              this.updateFirmwareVersionInfo();
            }
            // Status report
            else if (validPacket.packet[1] == gdl90.UAVIONIX_SIGNATURE &&
              validPacket.packet[2] == gdl90.UAVIONIX_OEM_STATUS_REPORT) {
              this.packetCounts.uavionixStatusReport++;
              var oldStatusReport = this.statusReport;
              // Select appropriate version configuration report
              if (validPacket.packet[3] == 0x01) {
                this.statusReport = gdl90.decodeStatusReport(validPacket.packet);
              } else if (validPacket.packet[3] >= 0x02) {
                this.statusReport = gdl90.decodeStatusReportV2(validPacket.packet);
              }
              this.statusReportReceived = true;
            }
            // Tower status report
            else if (validPacket.packet[1] == gdl90.UAVIONIX_SIGNATURE &&
              validPacket.packet[2] == gdl90.UAVIONIX_OEM_TOWER_STATUS_REPORT) {
              this.packetCounts.uavionixTowerStatusReport++;
            }
            // Device startup 
            else if (validPacket.packet[1] == gdl90.UAVIONIX_SIGNATURE &&
              validPacket.packet[2] == gdl90.UAVIONIX_OEM_DEVICE_STARTUP) {
              this.packetCounts.uavionixDeviceStartup++;
            }
          }

        }

        this.updateMonitorInfo();
        this.updatePacketCounter();

        if (this.connectionTimeout !== null) {
          clearTimeout(this.connectionTimeout);
          this.connectionTimeout = null;
        }
        // Wait 2 seconds before timing out device
        this.connectionTimeout = setTimeout(this.deviceDisconnected.bind(this), 2000);
      } else {
        this.invalidPackets++;
        this.updatePacketCounter();
        console.log("Received invalid packet");
      }
    }.bind(this)); // forEach packet
  },
  parseIntIfDisplayed: function (fieldName, defaultValue) {
    if (document.getElementById(fieldName + "Span").style.display == "none") {
      return defaultValue;
    } else {
      return parseInt(document.getElementById(fieldName).value, 10);
    }
  },
  validateDataAndSend: function () {
    if (ui.validateData()) {
      console.log("Data validated");

      // Find if we have any "setup" data to send
      // which is optional based on device type
      // Only use data if showing for this device
      var defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous;
      var setupSource, controlSources, gpsSource, displayOutput;
      var com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol, baroAltSource;

      defaultVfrCode = this.parseIntIfDisplayed("vfrCode", 0xFFFF);
      maxSpeed = this.parseIntIfDisplayed("maxSpeed", 0xFF);
      adsbIn = this.parseIntIfDisplayed("adsbIn", 0xFF);
      control = this.parseIntIfDisplayed("control", 0xFF);
      flightPlanId = document.getElementById("flightPlanId").value.trim();
      csidLogic = this.parseIntIfDisplayed("csidLogic", 0xFF);
      anonymous = this.parseIntIfDisplayed("anonymous", 0xFF);
      setupSource = this.parseIntIfDisplayed("setupSource", 0xFF);
      controlSources = this.parseIntIfDisplayed("controlSources", 0xFF);
      gpsSource = this.parseIntIfDisplayed("gpsSource", 0xFF);
      displayOutput = this.parseIntIfDisplayed("displayOutput", 0xFF);
      com1Rate = this.parseIntIfDisplayed("com1Rate", 0xFF);
      com1Data = this.parseIntIfDisplayed("com1Data", 0xFF);
      com1PHY = this.parseIntIfDisplayed("com1PHY", 0xFF);
      // Use same protocol in both directions currently
      com1InputProtocol = this.parseIntIfDisplayed("com1Protocol", 0xFFFF);
      com1OutputProtocol = this.parseIntIfDisplayed("com1Protocol", 0xFFFF);
      com2Rate = this.parseIntIfDisplayed("com2Rate", 0xFF);
      com2Data = this.parseIntIfDisplayed("com2Data", 0xFF);
      com2PHY = this.parseIntIfDisplayed("com2PHY", 0xFF);
      com2InputProtocol = this.parseIntIfDisplayed("com2InputProtocol", 0xFFFF);
      com2OutputProtocol = this.parseIntIfDisplayed("com2OutputProtocol", 0xFFFF);
      baroAltSource = this.parseIntIfDisplayed("baroAltSource", 0xFF);

      var gdl90Packet = gdl90.constructStatic().buffer;
      serial.send(gdl90Packet, function (sendResult) {
        // The setup packet is unreliable if we send back to back too quickly
        // Cringeworthy delay - 500ms
        setTimeout(function () {
          var gdl90SetupPacket = gdl90["constructSetupV" + ui.currentDevice.setupPacketVersion](0xFF, 0xFF, 0xFFFF, 0xFF, control, defaultVfrCode, maxSpeed, adsbIn, csidLogic, anonymous, flightPlanId, setupSource, controlSources, gpsSource, displayOutput, com1Rate, com1Data, com1PHY, com1InputProtocol, com1OutputProtocol, com2Rate, com2Data, com2PHY, com2InputProtocol, com2OutputProtocol, baroAltSource).buffer;
          serial.send(gdl90SetupPacket, function (sendResult) {
            // RCB TODO Wait for status response before notifying that the device was configured
            require('electron').remote.dialog.showMessageBox({
              type: "info",
              title: "Configured",
              message: "Device Configured"
            });
          });
        }, 500);
      });
    } else {
      console.log("Unable to validate data");
    }
  },
  validateAdvancedAndSend: function () {
    if (ui.validateAdvanced()) {
      var sda, sil, threshold, outputFormat;
      sda = this.parseIntIfDisplayed("sda", 0xFF);
      sil = this.parseIntIfDisplayed("sil", 0xFF);
      threshold = this.parseIntIfDisplayed("threshold", 0xFFFF);
      outputFormat = this.parseIntIfDisplayed("outputFormat", 0xFF);

      // Control, default VFR, max aircraft speed, and ADS-B IN capability are all DNC here, those are set from non-advanced
      var gdl90Packet = gdl90["constructSetupV" + ui.currentDevice.setupPacketVersion](sil, sda, threshold, outputFormat, 0xFF, 0xFFFF, 0xFF, 0xFF, 0xFF, 0xFF, null, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFFFF, 0xFFFF, 0xFF, 0xFF, 0xFF, 0xFFFF, 0xFFFF, 0xFF).buffer;
      serial.send(gdl90Packet, function (sendResult) {
        //navigator.notification.alert("Advanced Settings Configured", null, "Updated", "OK");
        require('electron').dialog.showMessageBox({
          type: "info",
          title: "Configured",
          message: "Advanced Settings Configured"
        });
      });
    } else {
      console.log("Unable to validate advanced data");
    }
  },
  loadGeofenceFile: function (file) {
    if (typeof device != "undefined" && device.platform == "Android") {
      return this.loadGeofenceFileAndroid(file);
    } else {
      return this.loadGeofenceFileOther(file);
    }
  },
  loadGeofenceFileAndroid: function (file) {
    var statusElem = document.getElementById("geofenceParseStatus");
    statusElem.textContent = "Opening file..";
    document.getElementById("geofenceButton").disabled = true;
    cordova.plugins.diagnostic.requestRuntimePermission(function (status) {
      if (status != cordova.plugins.diagnostic.permissionStatus.GRANTED) {
        statusElem.textContent = "You need to give this app permission to access storage.";
        return;
      }

      if (!file)
      {
        // No file selected
        statusElem.textContent = "";
        return;
      }

      var reader = new FileReader();
      reader.onloadend = function () {
        var kmlSrc = reader.result;
        try {
          this.geofenceData = null;
          this.geofenceData = geofence.loadPolygonDataFromKml(kmlSrc);
          statusElem.textContent = "Opened file successfully";

          document.getElementById("geofenceButton").disabled = false;
        } catch (ex) {
          statusElem.textContent = "Failed to open file: " + ex.message;
          document.getElementById("geofenceButton").disabled = true;
        }
      }.bind(this);
      reader.readAsText(file);
    }.bind(this), function (error) {
      statusElem.textContent = "You need to give this app permission to access storage.";
    }, cordova.plugins.diagnostic.permission.READ_EXTERNAL_STORAGE);
  },
  loadGeofenceFileOther: function (file) {
    var statusElem = document.getElementById("geofenceParseStatus");
    statusElem.textContent = "Opening file..";
    document.getElementById("geofenceButton").disabled = true;
    
    if (!file)
    {
      // No file selected
      statusElem.textContent = "";
      return;
    }

    var reader = new FileReader();
    reader.onloadend = function () {
      var kmlSrc = reader.result;
      try {
        this.geofenceData = null;
        this.geofenceData = geofence.loadPolygonDataFromKml(kmlSrc);
        statusElem.textContent = "Opened file successfully";

        document.getElementById("geofenceButton").disabled = false;
      } catch (ex) {
        statusElem.textContent = "Failed to open file: " + ex.message;
        document.getElementById("geofenceButton").disabled = true;
      }
    }.bind(this);
    reader.readAsText(file);
  },
  sendGeofenceData: function () {
    var sequence = 0;
    var packetType = gdl90.GEOFENCE_SUBTYPE_RESET;
    var i = 0;
    var timeout = null;
    var retries = 0;
    var MAX_RETRIES = 4;
    // 10 for header, CRC
    var MAX_POINTS_SIZE = Math.floor((256) / geofence.SIZE_OF_PACKED_POINT) *
      geofence.SIZE_OF_PACKED_POINT;
    var MAX_POLYGONS_SIZE = Math.floor((256) / geofence.SIZE_OF_PACKED_POLYGON) *
      geofence.SIZE_OF_PACKED_POLYGON;

    var statusElem = document.getElementById("geofenceParseStatus");

    if (this.geofenceData == null) {
      // Indicate error
      console.log("No geofence data available");
      statusElem.textContent = "No geofence file selected.";
      return;
    }
    else
      statusElem.textContent = "Uploading...";

    // `packet` has had the flag bytes and CRC removed but is otherwise complete.
    this.geofencePacketReceived = function (packet) {
      var statusElem = document.getElementById("geofenceParseStatus");
      
      var payload, packet, crc;
      if (packet) {
        clearInterval(timeout);
        timeout = null;
        if (packet[5] === gdl90.GEOFENCE_SUBTYPE_ACK) {
          console.log("Received ACK packet. Sequence: " + packet[4]);
          if (packet[6] != gdl90.GEOFENCE_ERROR_NONE)
            console.log("  Error code in packet: " + packet[6]);
        }
        if (packet[4] !== ((sequence - 1) & 0xFF))
          console.log("  Incorrect sequence number: " + packet[4]);
        if (packet[4] < ((sequence - 1) & 0xFF))
          return; // duplicate packet
      }

      // Dummy payload
      payload = new Uint8Array(0);

      if (packetType === gdl90.GEOFENCE_SUBTYPE_POINTS) {
        payload = new Uint8Array(2 + MAX_POINTS_SIZE);

        // Fetch a chunk of points data
        var points = this.geofenceData.points.subarray(i, i + MAX_POINTS_SIZE);
        i += points.length;

        // Load point count into length field
        var pointCount = points.length / geofence.SIZE_OF_PACKED_POINT;
        payload[0] = (pointCount) & 0xFF;
        payload[1] = (pointCount >> 8) & 0xFF;
        
        statusElem.textContent = "Uploading point " + (i / geofence.SIZE_OF_PACKED_POINT) + "/" + (this.geofenceData.points.length / geofence.SIZE_OF_PACKED_POINT) + "...";

        // Append points data to payload
        payload.set(points, 2);
      }
      else if (packetType === gdl90.GEOFENCE_SUBTYPE_POLYGONS) {
        payload = new Uint8Array(2 + MAX_POLYGONS_SIZE);

        // Fetch a chunk of poly data
        var polygons = this.geofenceData.polygons.subarray(i, i + MAX_POLYGONS_SIZE);
        i += polygons.length;

        // Load point count into length field
        var polyCount = polygons.length / geofence.SIZE_OF_PACKED_POLYGON;
        payload[0] = (polyCount) & 0xFF;
        payload[1] = (polyCount >> 8) & 0xFF;
        
        statusElem.textContent = "Uploading polygon " + (i / geofence.SIZE_OF_PACKED_POLYGON) + "/" + (this.geofenceData.polygons.length / geofence.SIZE_OF_PACKED_POLYGON) + "...";

        // Append poly data to payload
        payload.set(polygons, 2);
      }
      else if (packetType === gdl90.GEOFENCE_SUBTYPE_VERIFY) {
        payload = new Uint8Array(6);

        crc = crc32.calculateFromSeed(0x01700170, this.geofenceData.points);
        crc = crc32.calculateFromSeed(crc, this.geofenceData.polygons);

        payload[2] = crc & 0xFF;
        payload[3] = (crc >> 8) & 0xFF;
        payload[4] = (crc >> 16) & 0xFF;
        payload[5] = (crc >> 24) & 0xFF;
      }
      else if (packetType === null) {
        this.geofencePacketReceived = null;

        // Build device CRC from response
        var deviceCrc = packet[8];
        deviceCrc |= packet[9] << 8;
        deviceCrc |= packet[10] << 16;
        deviceCrc |= packet[11] << 24;

        // Calculate the local CRC
        crc = crc32.calculateFromSeed(0x01700170, this.geofenceData.points);
        crc = crc32.calculateFromSeed(crc, this.geofenceData.polygons);

        if (deviceCrc != crc) {
          console.log("CRC error. Got: " + deviceCrc + " Expected: " + crc);
          statusElem.textContent = "Failed to upload geofence data";
        }
        else
          statusElem.textContent = "Geofence uploaded successfully";

        console.log("Finished uploading geofence");
        return;
      }

      // Send geofence packet to device
      console.log("Sending packetType " + packetType + ", sequence " + sequence);
      packet = gdl90.constructGeofence(packetType, sequence, payload);
      serial.send(packet.buffer, null);

      retries = 0;
      timeout = setInterval(function () {
        if (retries < MAX_RETRIES) {
          serial.send(packet.buffer, null);
          ++retries;
          console.log("Resending geofence packet");
        } else {
          clearInterval(timeout);
          timeout = null;
          this.geofencePacketReceived = null;
          console.log("Failed to upload geofence (timed out)");

          // Grab status element for message
          var statusElem = document.getElementById("geofenceParseStatus");
          statusElem.textContent = "Failed to upload geofence data";
        }
      }.bind(this), 500);

      if (packetType === gdl90.GEOFENCE_SUBTYPE_RESET) {
        packetType = gdl90.GEOFENCE_SUBTYPE_POINTS;
      } else if (packetType === gdl90.GEOFENCE_SUBTYPE_POINTS && i >= this.geofenceData.points.length) {
        packetType = gdl90.GEOFENCE_SUBTYPE_POLYGONS;
        i = 0;
      } else if (packetType === gdl90.GEOFENCE_SUBTYPE_POLYGONS && i >= this.geofenceData.polygons.length) {
        packetType = gdl90.GEOFENCE_SUBTYPE_VERIFY;
        i = 0;
      } else if (packetType === gdl90.GEOFENCE_SUBTYPE_VERIFY) {
        packetType = null; // no more packets to send
      }
      sequence = (sequence + 2) & 0xFF;
    };

    this.geofencePacketReceived(null);
  },
  resetGeofenceData: function() {
    var statusElem = document.getElementById("geofenceParseStatus");
    var resetTimeout = null;
    // Only reset if we aren't updating
    if (this.geofencePacketReceived == null) {

      // `packet` has had the flag bytes and CRC removed but is otherwise complete.
      this.geofencePacketReceived = function (packet) {
        if (packet) {
          if (packet[5] === gdl90.GEOFENCE_SUBTYPE_ACK) {
            if (resetTimeout) {
              clearTimeout(resetTimeout);
              resetTimeout = null;
            }
            console.log("Received reset ACK packet");
            //navigator.notification.alert("Geofence Reset", null, "Geofence", "OK");
            statusElem.textContent = "Geofence successfully reset";
            this.geofencePacketReceived = null;
          }
        }
      };
      resetTimeout = setTimeout(function () {
        this.geofencePacketReceived = null;
        console.log("Failed to reset geofence data (timed out)");
        //navigator.notification.alert("Geofence Reset Failed", null, "Geofence", "OK");
        statusElem.textContent = "Geofence reset failed";
        resetTimeout = null;
      }.bind(this), 500);
      packet = gdl90.constructGeofence(gdl90.GEOFENCE_SUBTYPE_RESET, 0, []);
      serial.send(packet.buffer, null);
    } else {
      // Unable to reset, another operation in progress
      console.log("Unable to reset geofence, another operation in progress");
      navigator.notification.alert("Unable to reset Geofence, operation in progress", null, "Geofence", "OK");
    }
  },
  onLogoClickTimeout: function () {
    this.logoClicked = 0;
  },
  onLogoClick: function (clickEvent) {
    // Click three times to hide or display counts
    this.logoClicked++;
    if (this.logoClicked >= 3) {
      if (this.logoClickedTimeout !== null) {
        clearTimeout(this.logoClickedTimeout);
        this.logoClickedTimeout = null;
      }
      this.logoClicked = 0;
      this.advancedDisplayed = !this.advancedDisplayed;
      document.getElementById("transmitControls").setAttribute('style', this.advancedDisplayed ? 'display:block;' : 'display:none;');
      document.getElementById("advancedConfig").setAttribute('style', this.advancedDisplayed ? 'display:block;' : 'display:none;');
      document.getElementById("advancedMonitor").setAttribute('style', this.advancedDisplayed ? 'display:block;' : 'display:none;');
      document.getElementById("advancedCommon").setAttribute('style', this.advancedDisplayed ? 'display:block;' : 'display:none;');
    } else {
      if (this.logoClickedTimeout !== null) {
        clearTimeout(this.logoClickedTimeout);
        this.logoClickedTimeout = null;
      }
      this.logoClickedTimeout = setTimeout(this.onLogoClickTimeout.bind(this), 1000);
    }
  },
  updatePacketCounter: function () {
    document.getElementById("validPackets").innerHTML = this.validPackets;
    document.getElementById("invalidPackets").innerHTML = this.invalidPackets;
    document.getElementById("heartbeatPackets").innerHTML = this.packetCounts.heartbeat;
    document.getElementById("initPackets").innerHTML = this.packetCounts.init;
    document.getElementById("ownshipPackets").innerHTML = this.packetCounts.ownship;
    document.getElementById("ownshipGeoAltPackets").innerHTML = this.packetCounts.ownshipGeoAlt;
    document.getElementById("hatPackets").innerHTML = this.packetCounts.hat;
    document.getElementById("uplinkPackets").innerHTML = this.packetCounts.uplink;
    document.getElementById("trafficPackets").innerHTML = this.packetCounts.traffic;
    document.getElementById("basicPackets").innerHTML = this.packetCounts.basic;
    document.getElementById("longPackets").innerHTML = this.packetCounts.long;
    document.getElementById("uavionixConfigReportPackets").innerHTML = this.packetCounts.uavionixConfigReport;
    document.getElementById("uavionixStatusReportPackets").innerHTML = this.packetCounts.uavionixStatusReport;
    document.getElementById("uavionixTowerStatusReportPackets").innerHTML = this.packetCounts.uavionixTowerStatusReport;
    document.getElementById("uavionixDeviceStartupPackets").innerHTML = this.packetCounts.uavionixDeviceStartup;
    document.getElementById("uavionixGeofencePackets").innerHTML = this.packetCounts.uavionixGeofence;
  },
  updateFirmwareVersionInfo: function () {
    if (this.configReportReceived) {
      document.getElementById("firmwareVersions").innerHTML = "Device HW / FW ID: " + this.configReport.adsbHwId + " / " + this.configReport.adsbFwId + "<br>Device Version: " + this.configReport.adsbSwVersionMajor + "." + this.configReport.adsbSwVersionMinor + "." + this.configReport.adsbSwVersionRevision;
    } else {
      document.getElementById("firmwareVersions").innerHTML = "Device Version: Unknown";
    }
  },
  updateMonitorInfo: function () {
    if (this.ownshipReceived) {
      document.getElementById("addressMonitor").innerHTML = this.ownship.address;
      document.getElementById("callsignMonitor").innerHTML = this.ownship.callsign;
      if (this.statusReportReceived) {
        document.getElementById("flightplanMonitor").innerHTML = this.statusReport.currentFlightplan;
      } else {
        document.getElementById("flightplanMonitor").innerHTML = "--";
      }
      document.getElementById("emitterCategoryMonitor").innerHTML = this.ownship.emitterCategoryName;
      if (this.ownship.latitude === "--" && this.ownship.longitude === "--") {
        document.getElementById("latitudeMonitor").innerHTML = this.ownship.latitude;
        document.getElementById("longitudeMonitor").innerHTML = this.ownship.longitude;
      } else if (ui.latLonFormat === "dms") {
        // Convert to DMS
        var point = new GeoPoint(this.ownship.longitude, this.ownship.latitude);
        document.getElementById("latitudeMonitor").innerHTML = point.getLatDeg();
        document.getElementById("longitudeMonitor").innerHTML = point.getLonDeg();
      } else {
        document.getElementById("latitudeMonitor").innerHTML = this.ownship.latitude.toFixed(5) + "&deg;";
        document.getElementById("longitudeMonitor").innerHTML = this.ownship.longitude.toFixed(5) + "&deg;";
      }
      if (this.ownship.altitude === "--") {
        document.getElementById("altitudePressureMonitor").innerHTML = this.ownship.altitude;
      } else if (ui.altitudeUnits === "ft") {
        document.getElementById("altitudePressureMonitor").innerHTML = this.ownship.altitude + " ft";
      } else {
        document.getElementById("altitudePressureMonitor").innerHTML = (this.ownship.altitude * 0.3048).toFixed(1) + " m";
      }
      if (this.ownshipGeoAltReceived && this.ownship.latitude != "--") {
        if (ui.altitudeUnits === "ft") {
          document.getElementById("altitudeGpsMonitor").innerHTML = this.ownshipGeoAlt.altitude + " ft";
        } else {
          document.getElementById("altitudeGpsMonitor").innerHTML = (this.ownshipGeoAlt.altitude * 0.3048).toFixed(1) + " m";
        }
      } else {
        document.getElementById("altitudeGpsMonitor").innerHTML = "--";
      }
      document.getElementById("nicMonitor").innerHTML = this.ownship.NIC;
      document.getElementById("nacpMonitor").innerHTML = this.ownship.NACp;
      document.getElementById("emergencyMonitor").innerHTML = this.ownship.emergencyName;
    } else {
      // No ownship info received
      document.getElementById("addressMonitor").innerHTML = "--";
      document.getElementById("callsignMonitor").innerHTML = "--";
      document.getElementById("flightplanMonitor").innerHTML = "--";
      document.getElementById("emitterCategoryMonitor").innerHTML = "--";
      document.getElementById("latitudeMonitor").innerHTML = "--";
      document.getElementById("longitudeMonitor").innerHTML = "--";
      document.getElementById("altitudePressureMonitor").innerHTML = "--";
      document.getElementById("altitudeGpsMonitor").innerHTML = "--";
      document.getElementById("nicMonitor").innerHTML = "--";
      document.getElementById("nacpMonitor").innerHTML = "--";
      document.getElementById("emergencyMonitor").innerHTML = "--";
    }
    if (this.configReportReceived) {
      //document.getElementById("advancedConfigMonitor").innerHTML = "<br><span class='monitorField'>SIL:</span> " + this.configReport.sil;
      //document.getElementById("advancedConfigMonitor").innerHTML += "<br><span class='monitorField'>SDA:</span> " + this.configReport.sda;
      //document.getElementById("advancedConfigMonitor").innerHTML += "<br><span class='monitorField'>Output Format:</span> " + this.configReport.outputFormat;
      //document.getElementById("advancedConfigMonitor").innerHTML += "<br><span class='monitorField'>Sniffer Threshold:</span> " + this.configReport.sniffThreshold;
    }
  },
  waitForOwnshipAndStatus: function () {
    // We have received only an ownship or status packet
    // That means we have only partial configuration information - load and let the user know
    this.ownshipToStatusTimer = null;
    this.loadConfiguration();
  },
  // Load configuration data received in ownship and status messages
  loadConfiguration: function () {
    if (!this.ownshipReceived && !this.configReportReceived) {
      //navigator.notification.alert("No configuration information available from device. To view existing configuration please ensure GPS can obtain a lock.", null, "Configuration", "OK");
      // We will get here if no device is connected
    } else {
      // Crossfill ownship data to configuration screen
      if (this.ownshipReceived) {
        if (!(this.configReportReceived) || (this.configReport.messageVersion < 3)) {
          // If we don't ahve the default callsign and ICAO in the configuration report, use
          // what we have in the ownship
          document.getElementById("icao").value = this.ownship.address.trim();
          document.getElementById("callsign").value = this.ownship.callsign.trim();
        }
        document.getElementById("emitter").value = this.ownship.emitterCategory;
      } else {
        //if (!this.deliveredConfigurationWarning) {
        //  navigator.notification.alert("Configuration information not fully received from device. To view existing configuration please ensure GPS can obtain a lock.", null, "Configuration", "OK");
        //  this.deliveredConfigurationWarning = true;
        //}
      }
      // If status info available, crossfill that data too
      if (this.configReportReceived) {
        var receivedDevice = ui.deviceIdToDevice(this.configReport.adsbFwId);
        if (receivedDevice != null) {
          document.getElementById("deviceType").value = receivedDevice.id;
          document.getElementById("deviceType").disabled = true;
        } else {
          document.getElementById("deviceType").disabled = false;
        }
        document.getElementById("emitter").value = this.configReport.emitterCategory;
        // Fill all fields regardless of device selection
        document.getElementById("Vs0").value = this.configReport.stallSpeed;
        // Fill length then width (dynamically generated)
        document.getElementById("aLength").value = this.configReport.aLength;
        ui.populateWidths();
        document.getElementById("aWidth").value = this.configReport.aWidth;
        document.getElementById("offsetLat").value = this.configReport.antOffsetLat;
        document.getElementById("offsetLon").value = this.configReport.antOffsetLon;
        document.getElementById("sil").value = this.configReport.sil;
        document.getElementById("sda").value = this.configReport.sda;
        document.getElementById("outputFormat").value = this.configReport.outputFormat;
        // txControl is a bit special in that it is very specific to the device type
        // Try to set here but if it doesn't match the device type it will be "fixed up"
        // in deviceTypeUpdated()
        document.getElementById("control").value = this.configReport.txControl;
        document.getElementById("threshold").value = this.configReport.sniffThreshold;
        document.getElementById("vfrCode").value = this.configReport.defaultVfrCode;
        document.getElementById("maxSpeed").value = this.configReport.maxSpeed;
        document.getElementById("adsbIn").value = this.configReport.adsbIn;
        //this.setView("configuration");

        if (this.configReport.messageVersion >= 2) {
          document.getElementById("csidLogic").value = this.configReport.csidLogic;
          document.getElementById("anonymous").value = this.configReport.anonymous;
          document.getElementById("flightPlanId").value = this.configReport.flightPlanId.trim();
          document.getElementById("setupSource").value = this.configReport.setupSource;
          document.getElementById("controlSources").value = this.configReport.controlSources;
          document.getElementById("gpsSource").value = this.configReport.gpsSource;
          document.getElementById("displayOutput").value = this.configReport.displayOutput;
          document.getElementById("com1Rate").value = this.configReport.com1Rate;
          document.getElementById("com1Data").value = this.configReport.com1Data;
          document.getElementById("com1PHY").value = this.configReport.com1PHY;
          // We have input/output protocols, map input for our singular UI element
          document.getElementById("com1Protocol").value = this.configReport.com1InputProtocol;
          document.getElementById("com2Rate").value = this.configReport.com2Rate;
          document.getElementById("com2InputProtocol").value = this.configReport.com2InputProtocol;
          document.getElementById("com2OutputProtocol").value = this.configReport.com2OutputProtocol;
        }

        if (this.configReport.messageVersion >= 3) {
          // Use configuration report ICAO and callsign instead of ownship
          document.getElementById("icao").value = this.configReport.address.trim();
          document.getElementById("callsign").value = this.configReport.callsign.trim();
          document.getElementById("baroAltSource").value = this.configReport.baroAltSource;
        }
      } else {
        //this.setView("configuration");
        if (!this.deliveredConfigurationWarning) {
          //navigator.notification.alert("Connected device has limited ability to display the programmed configuration. The current address, callsign, and emitter are shown. Please configure all other fields.", null, "Configuration", "OK");
          this.deliveredConfigurationWarning = true;
        }
      }
    }
    // deviceTypeUpdated forces recomputation of defaults in case the configuration doesn't match the existing device
    // (e.g. the control bits)
    if (this.configReport != null) {
      ui.deviceTypeUpdated(this.configReport.txControl);
    } else {
      ui.deviceTypeUpdated(document.getElementById("control").value);
    }
  },
  connectComPort: function() {
    // Connect COM port
    serial.connectPort(document.getElementById("comPort").value);
  },
  resetPacketCounts: function () {
    this.packetCounts.heartbeat = 0;
    this.packetCounts.init = 0;
    this.packetCounts.uplink = 0;
    this.packetCounts.hat = 0;
    this.packetCounts.ownship = 0;
    this.packetCounts.ownshipGeoAlt = 0;
    this.packetCounts.traffic = 0;
    this.packetCounts.uavionixConfigReport = 0;
    this.packetCounts.uavionixStatusReport = 0;
    this.packetCounts.uavionixTowerStatusReport = 0;
    this.packetCounts.uavionixDeviceStartup = 0;
    this.packetCounts.uavionixGeofence = 0;
    this.validPackets = 0;
    this.invalidPackets = 0;
  },
  toggleDevTools() {
    require('electron').remote.getCurrentWindow().webContents.toggleDevTools();
  }
};
